'use client';

import PageHeader from "@/components/layout/Pages/PageHeader";
import { useLocale } from 'next-intl';
import PageItem from "./components/PageItem";
import HeaderSection from "./components/HeaderSection";
import MediaItem from "./components/MediaItem";
import { useEffect, useState } from "react";

interface LocalizedText {
  ar: string;
  en: string;
}

interface SEOData {
  title: LocalizedText;
  description: LocalizedText;
}

interface PageMain {
  id: number;
  title: LocalizedText;
  description: LocalizedText;
  cover: string | null;
  type: string;
}

interface Section {
  id: number;
  title: LocalizedText;
  description: LocalizedText;
  media: string;
  type: string;
}

interface ContentProps {
  seo: SEOData;
  pageMain: PageMain;
  sections: Section[];
}

export default function Content({ seo, pageMain, sections }: ContentProps) {
  const locale = useLocale() as 'ar' | 'en';
  const [currentAssets, setCurrentAssets] = useState<number>(0);
  const [data, setdata] = useState<Section | undefined>(undefined);

  useEffect(() => {
    const foundData = sections.find(a => a.id === currentAssets);
    setdata(foundData);
  }, [currentAssets, sections]);

  const PageContent = ({ type }: { type: string }) => {
    if (type === "media") {
      const hasTitle = !!data?.title?.[locale] && data.title[locale].length > 0;
      const hasDescription = !!data?.description?.[locale] && data.description[locale].length > 0;

      return (
        <div className="w-full flex flex-col gap-2 md:flex-row">
          <div className="flex h-fit flex-row flex-1 flex-wrap gap-2">
            {sections.map((section) => (
              <MediaItem 
                key={section.id} 
                mediaAction={setCurrentAssets} 
                active={section.id === currentAssets} 
                id={section.id} 
                title={section.title} 
                description={section.description} 
                media={section.media} 
                type={section.type} 
              />
            ))}
          </div>
          
          <div className="flex h-fit flex-row flex-1 flex-wrap gap-2">
            {currentAssets > 0 && data && (
              <div className="flex w-full flex-col gap-4 bg-gray-50 rounded-2xl overflow-hidden border border-gray-100">
                <div className="p-4">
                  <img 
                    className="w-full h-auto rounded-lg" 
                    src={data.media} 
                    alt={data.title[locale] || "Media item"} 
                    title={data.title[locale]} 
                  />
                </div>
                
                {(hasTitle || hasDescription) && (
                  <div className="w-full p-4 rounded-b-2xl flex flex-col gap-2 bg-gray-100">
                    {hasTitle && (
                      <div className="w-full flex align-middle justify-center text-center font-almarai text-[18px] text-main-blue/90 font-bold">
                        {data.title[locale]}
                      </div>
                    )}
                    {hasDescription && (
                      <div className="w-full flex align-middle justify-center text-center font-almarai font-light text-[14px] text-gray-500">
                        {data.description[locale]}
                      </div>
                    )}
                  </div>
                )}
              </div>
            )}
          </div>
        </div>
      );
    } else {
      return (
        <>
          {sections.map((section) => (
            <PageItem 
              key={section.id} 
              id={section.id} 
              title={section.title} 
              description={section.description} 
            />
          ))}
        </>
      );
    }
  };

  return (
    <>
      <PageHeader
        title={seo.title[locale]}
        description={seo.description[locale]}
      />

      <div className='h-full w-full flex flex-col justify-start items-center gap-0'>
        <HeaderSection 
          id={pageMain.id} 
          cover={pageMain.cover} 
          title={pageMain.title} 
          description={pageMain.description} 
        />

        <div className="container w-full flex flex-col justify-start items-start px-4 md:px-0 mt-10 mb-20 gap-6">
          <div className="w-full flex flex-col justify-start items-start gap-2">
            <PageContent type={pageMain.type} />
          </div>
        </div>
      </div>
    </>
  );
}